//Semi-persistent data storage for sensor-interface.html
//Two databases exist:
//1. configDB: 	the configuration/settings database
//2. dataDB: 	the measurement data database

// -- configDB --
// stores user configuration data (options pane) and calibration data (per board)
// structure:
// configDB
//  |
//  |---global [objectStore]
//  |      |
//  |      |--framerate    INT
//  |      |--autoconnect  BOOL
//  |      |--
//  |
//  |---realtime_graph [objectStore]
//  |           |
//  |           |---background HEX6
//  |           |---subgraph   INT
//  |           |---(etc. - see sgraph_options object in defaults.js)
//  |
//  |---fft_graph [objectStore]
//  |
//  |---calibration [objectStore]
//          |
//          |---[board name]
//			|     |
//			|	  |---channel 1 array
//          :     |---channel 2 array (etc.)


// -- dataDB --
// stores rawdata and slowdata as it is being purged by the garbage collector (roughly every 5 seconds)
// structure:
// dataDB
//  |
//  |--slow_data [objectStore]
//	|     |
//  |     |-- name: [start-timecode], data: slowdata
//  |  
//  |--fast_data
//        |
//        |-- name [start-timecode], data: fast_data

var databaseOpen = false;
var dataDB;
var configDB;

function openDatabase(){
	if (!window.indexedDB) {
		window.alert("Your browser doesn't support a stable version of IndexedDB. You will not be able to save or download measurement data.");
		return;
	}
	if(!databaseOpen){
		dbSupported = true;
		databaseOpen = true;
		
		var req = window.indexedDB.open('dataDB',DB_VERSION);
		
		req.onerror 		= function(e){
			alert("Error opening indexedDB for data store: " + e.message);
		}
		req.onupgradeneeded = function(event){
			dataDB = event.target.result;
			dataDB.createObjectStore("fastData",{keyPath:"i"});
			dataDB.createObjectStore("slowData",{keyPath:"i"});
			dataDB.oncomplete = function(){createDataObjectIndex();}	
		}
		req.onsuccess  		= function(event){
			dataDB = event.target.result;
			createDataObjectIndex(dataDB);
		}
		
		var req2 = window.indexedDB.open('configDB', DB_VERSION);
		
		req2.onupgradeneeded = function(event){
			configDB = event.target.result;
			configDB.createObjectStore("calibration",{keyPath:"board"});
			configDB.createObjectStore("global",{keyPath:"property"});
			configDB.createObjectStore("realtime_graph",{keyPath:"property"});
			configDB.createObjectStore("calibration_graph",{keyPath:"property"});
			configDB.createObjectStore("fft_graph",{keyPath:"property"});
			configDB.createObjectStore("review",{keyPath:"property"});
			configDB.oncomplete = function(){insertDefaults();}
		}
		
		req2.onsuccess = function(event){
			configDB = event.target.result;
			loadConfigData();
		}
	}
}

function createDataObjectIndex(){
	//dataDB	
	var transaction = dataDB.transaction('fastData','readonly');
	var fastDataStore = transaction.objectStore('fastData');
	var cursorRequest = fastDataStore.openCursor();
	cursorRequest.onsuccess = function(e){
		if(e.target.result){
			addIndex(e.target.result.key,e.target.result.value.data[MIN_STORED_DECIMATION][3].length);
			e.target.result.continue();
		} else {	//we're done
			dbLoaded = true;
		}
	}
}

function addIndex(sessionName, vlength){
	sessionDate = new Date(); sessionDate.setTime(parseInt(sessionName));

	objStores[objStores.length] = [];
	objStores[objStores.length - 1][0] = sessionName;
	objStores[objStores.length - 1][1] = sessionDate;
	objStores[objStores.length - 1][2] = toOoMString(vlength);
}

var globalD;

function storeFastData(slice, sessionID){
	if(dbLoaded){
		var s = dataDB.transaction(["fastData"],"readwrite").objectStore("fastData");
		var fdreq = s.get(sessionID);
		fdreq.onsuccess = function(e){
			var d = fdreq.result;
			for(i = MIN_STORED_DECIMATION; i < slice.length; i++){
				j = slice[i][0];
				d.data[i][1] = slice[i][1];
				k = 0;
				while(j < slice[i][1]){
					d.data[i][3][j] = slice[i][3][k];
					j++; k++;
				}
			}
			s.put(d);
			
			//store this information in objStores
			var t = 0;
			while(objStores[t++][0] != sessionID){if(t == objStores.length) break;}
			objStores[t - 1][2] = toOoMString(d.data[MIN_STORED_DECIMATION][3].length);
		}
	}
}

function storeCalibrationData(slice, board){
	if(dbLoaded){
		var s = configDB.transaction(["calibration"],"readwrite").objectStore("calibration");
		var calreq = s.get(board);
		calreq.onsuccess = function(e){
			var d = calreq.result;
			//we now have in d the calibration data currently stored
			//and slice contains the 2-dimensional array we want to replace it with
			//so let's get going
			for(var i in slice){
				d[i] = new Array(); //whether the key exists or not, we either need to clear it or create it, both are done with the same statement
				for(var j = 0; j < slice[j].length; j++){ d[i][j] = slice[i][j]; } //push changes
			}
			
			s.put(d);	//push into configDB
		}	
	}
}

function readCalibrationData(board, returnfunction){
	if(dbLoaded){
		var s = configDB.transaction(["calibration"],"read").objectStore("calibration");
		var calreq = s.get(board);
		calreq.onsuccess = function(e){
			returnfunction(calreq.result);
		}	
	}
}

function storeSlowData(slice, sessionID){

}

//creates new index in the objectstore 
//todo: check if object exists
function dbNewSession(sessionID, fastDataPrototype, slowDataPrototype){

	t = dataDB.transaction(["fastData"],"readwrite").objectStore("fastData");
	t.add({i: sessionID, data: fastDataPrototype});
	t = dataDB.transaction(["slowData"],"readwrite").objectStore("slowData");
	t.add({i: sessionID, data: slowDataPrototype});
	/*t.onerror = function(){
		alert("error during transaction");
	}*/
}

function dbRemoveSession(sessionID){
	var req = dataDB.transaction(["fastData"],"readwrite").objectStore("fastData").delete(parseInt(sessionID));
	req.onsuccess = function(e){
		objStores = [];
		createDataObjectIndex();
		setTimeout('updateReviewList(true)',250);
	}
	req.onerror = function(e){
		alert("Error removing record: " + e.message);
	}
}

function toOoMString(val){
	var prefix = "";
	var div = 1;
	if		(val > 1000000000000)	{ prefix = "T"; div = 1000000000000;}
	else if	(val > 1000000000)		{ prefix = "G"; div = 1000000000;}
	else if (val > 1000000)			{ prefix = "M"; div = 1000000;}
	else if (val > 1000)			{ prefix = "k"; div = 1000;}
	val /= div;
	return (Math.round(val * 100)/100) + " " + prefix + "pts";
}

function downloadSession(sessionID){
	sessionID = parseInt(sessionID);
	var req = dataDB.transaction(["fastData"],"readwrite").objectStore("fastData").get(sessionID);
	req.onsuccess = function(e){
		csvFormattedDB = "Current data (raw)\n";
		for(i = 0; i < decimationArray.length; i++){
			csvFormattedDB += Math.round(100*MAX_SAMPLE_RATE/decimationArray[i])/100 + " samples/s,";
		}
		csvFormattedDB += "\n";
		
		var f = req.result.data;
		
		var maxlen = 0;
		for(i = 0; i < decimationArray.length; i++){
			if(f[i][3].length > maxlen) maxlen = f[i][3].length;
		}
		
		for(i = 0; i < maxlen; i++){
			for(j = 0; j < decimationArray.length; j++){
				if(i < f[j][3].length) csvFormattedDB += f[j][3][i];
				csvFormattedDB += ",";
			}
			csvFormattedDB +=  "\n";
		}
		
		var sessionDate = new Date(sessionID);
		var filename = sessionDate.getFullYear() + "_" + sessionDate.getMonth() + "_" + sessionDate.getDate() + " " + sessionDate.getHours() + "-" + sessionDate.getMinutes() + "-" + sessionDate.getSeconds() + ".csv";
		
		var a = document.createElement('a');
		a.setAttribute('href', URL.createObjectURL(new Blob([csvFormattedDB],{type: 'text/csv'})));
		a.setAttribute('download', filename);

		if (document.createEvent) {
			var event = document.createEvent('MouseEvents');
			event.initEvent('click', true, true);
			a.dispatchEvent(event);
		}
		else {
			a.click();
		}
	}
}

//by default, the defaults (duh) are loaded in through defaults.js. If other values have been stored in indexedDB, we load those in
function loadConfigData(){

}

//this function loads the default values in defaults.js into the config database
function insertDefaults(){
	
}
